<?php
/**
 * Script Universal untuk membuat template dari folder manapun
 * Usage: php create_template_from_folder.php <folder_path> <template_name>
 * 
 * Contoh:
 * php create_template_from_folder.php ../../secure-3 netflix
 */

require_once '../config/database.php';
require_once '../config/config.php';

// Get parameters
$sourceDir = isset($argv[1]) ? $argv[1] : '';
$templateName = isset($argv[2]) ? $argv[2] : '';

if (empty($sourceDir) || empty($templateName)) {
    die("Usage: php create_template_from_folder.php <source_folder> <template_name>\n" .
        "Example: php create_template_from_folder.php ../../secure-3 netflix\n");
}

// Convert relative path to absolute
if (!file_exists($sourceDir)) {
    $sourceDir = __DIR__ . '/' . $sourceDir;
}

$outputDir = __DIR__ . '/../templates/';
$zipFileName = $templateName . '.zip';

echo "Creating Template: $templateName\n";
echo "Source: $sourceDir\n\n";

// Check if source directory exists
if (!is_dir($sourceDir)) {
    die("Error: Source directory not found: $sourceDir\n");
}

// Create templates directory if not exists
if (!is_dir($outputDir)) {
    mkdir($outputDir, 0755, true);
    echo "Created templates directory.\n";
}

// Create ZIP file
$zip = new ZipArchive();
$zipPath = $outputDir . $zipFileName;

if ($zip->open($zipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
    die("Error: Cannot create ZIP file\n");
}

// Function to add files to ZIP recursively
function addDirectoryToZip($dir, $zip, $baseDir = '') {
    $files = scandir($dir);
    $excludedDirs = ['logs', '.git', 'node_modules', '.vscode', '__pycache__'];
    $excludedFiles = ['.gitignore', '.DS_Store', 'Thumbs.db'];
    
    foreach ($files as $file) {
        if ($file == '.' || $file == '..') {
            continue;
        }
        
        // Skip excluded directories
        if (is_dir($dir . '/' . $file) && in_array($file, $excludedDirs)) {
            continue;
        }
        
        // Skip excluded files
        if (in_array($file, $excludedFiles)) {
            continue;
        }
        
        $filePath = $dir . '/' . $file;
        $zipPath = $baseDir ? $baseDir . '/' . $file : $file;
        
        if (is_dir($filePath)) {
            // Add directory
            $zip->addEmptyDir($zipPath);
            // Recursively add files in directory
            addDirectoryToZip($filePath, $zip, $zipPath);
        } else {
            // Add file
            $zip->addFile($filePath, $zipPath);
        }
    }
}

// Add all files to ZIP
echo "Adding files to ZIP...\n";
addDirectoryToZip($sourceDir, $zip);

// Close ZIP
$zip->close();

$fileSize = filesize($zipPath);
echo "✓ Template ZIP created: $zipPath\n";
echo "  File size: " . number_format($fileSize / 1024 / 1024, 2) . " MB\n\n";

// Add to database
$conn = getDBConnection();

// Check if template already exists
$check = $conn->prepare("SELECT id FROM templates WHERE folder_name = ?");
$check->bind_param("s", $templateName);
$check->execute();
$result = $check->get_result();

if ($result->num_rows > 0) {
    echo "Template already exists in database. Updating...\n";
    $stmt = $conn->prepare("UPDATE templates SET zip_file_path = ?, is_active = 1 WHERE folder_name = ?");
    $stmt->bind_param("ss", $zipFileName, $templateName);
} else {
    echo "Adding template to database...\n";
    $name = ucfirst($templateName) . " Template";
    $description = "Template created from folder: $sourceDir";
    $stmt = $conn->prepare("INSERT INTO templates (name, folder_name, description, zip_file_path, is_active) VALUES (?, ?, ?, ?, 1)");
    $stmt->bind_param("ssss", $name, $templateName, $description, $zipFileName);
}

if ($stmt->execute()) {
    echo "✓ Template successfully added to database!\n";
    echo "\n✅ Template is ready to use!\n";
    echo "   You can now add clients and assign this template.\n";
} else {
    echo "✗ Error adding template to database: " . $conn->error . "\n";
}

$conn->close();

echo "\nDone!\n";
?>

