<?php
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/config.php';
requireLogin();

function formatBytes($bytes, $precision = 2) {
    $units = array('B', 'KB', 'MB', 'GB', 'TB');
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= pow(1024, $pow);
    return round($bytes, $precision) . ' ' . $units[$pow];
}

function getDiskSpace() {
    $total = disk_total_space(__DIR__);
    $free = disk_free_space(__DIR__);
    $used = $total - $free;
    return [
        'total' => $total,
        'free' => $free,
        'used' => $used,
        'percent' => $total > 0 ? round(($used / $total) * 100, 2) : 0
    ];
}

$diskSpace = getDiskSpace();
$conn = getDBConnection();

$dbStatus = 'connected';
$dbError = '';
try {
    $conn->query("SELECT 1");
} catch (Exception $e) {
    $dbStatus = 'error';
    $dbError = $e->getMessage();
}

$dbVersion = '';
$dbSize = 0;
if ($dbStatus === 'connected') {
    $result = $conn->query("SELECT VERSION() as version");
    if ($result) {
        $row = $result->fetch_assoc();
        $dbVersion = $row['version'];
    }
    
    $result = $conn->query("SELECT 
        ROUND(SUM(data_length + index_length) / 1024 / 1024, 2) AS 'size_mb'
        FROM information_schema.tables 
        WHERE table_schema = DATABASE()");
    if ($result) {
        $row = $result->fetch_assoc();
        $dbSize = $row['size_mb'] ?? 0;
    }
}

$requiredExtensions = ['mysqli', 'zip', 'json', 'curl', 'openssl'];
$extensions = [];
foreach ($requiredExtensions as $ext) {
    $extensions[$ext] = extension_loaded($ext);
}

$phpVersion = PHP_VERSION;
$phpMemoryLimit = ini_get('memory_limit');
$phpMaxUpload = ini_get('upload_max_filesize');
$phpMaxPost = ini_get('post_max_size');
$phpMaxExecution = ini_get('max_execution_time');

$serverSoftware = $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown';
$serverOS = PHP_OS;
$serverTime = date('Y-m-d H:i:s');
$timezone = date_default_timezone_get();

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>System Status - 405 Admin</title>
    <link rel="stylesheet" href="assets/css/admin.css">
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <style>
        .status-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        .status-card {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: var(--shadow);
        }
        .status-card h3 {
            color: var(--text-primary);
            font-size: 1rem;
            margin-bottom: 1rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        .status-item {
            display: flex;
            justify-content: space-between;
            padding: 0.75rem 0;
            border-bottom: 1px solid var(--border-color);
        }
        .status-item:last-child {
            border-bottom: none;
        }
        .status-item label {
            color: var(--text-secondary);
            font-size: 0.875rem;
        }
        .status-item value {
            color: var(--text-primary);
            font-weight: 500;
            font-size: 0.875rem;
        }
        .status-badge {
            display: inline-block;
            padding: 0.25rem 0.75rem;
            border-radius: 6px;
            font-size: 0.75rem;
            font-weight: 600;
        }
        .status-badge.success {
            background: rgba(74, 222, 128, 0.2);
            color: #4ade80;
            border: 1px solid rgba(74, 222, 128, 0.3);
        }
        .status-badge.error {
            background: rgba(255, 51, 102, 0.2);
            color: #ff3366;
            border: 1px solid rgba(255, 51, 102, 0.3);
        }
        .status-badge.warning {
            background: rgba(255, 170, 0, 0.2);
            color: #ffaa00;
            border: 1px solid rgba(255, 170, 0, 0.3);
        }
        .progress-bar {
            width: 100%;
            height: 8px;
            background: var(--bg-darker);
            border-radius: 4px;
            overflow: hidden;
            margin-top: 0.5rem;
        }
        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, var(--accent-cyan), var(--accent-teal));
            transition: width 0.3s;
        }
        .progress-fill.warning {
            background: linear-gradient(90deg, #ffaa00, #ff8800);
        }
        .progress-fill.danger {
            background: linear-gradient(90deg, #ff3366, #ff1155);
        }
    </style>
</head>
<body>
    <div class="admin-header">
        <h1>System Status</h1>
        <div class="user-info">
            <a href="index.php" class="btn btn-primary btn-sm">Dashboard</a>
            <a href="logout.php" class="btn btn-danger btn-sm">Logout</a>
        </div>
    </div>
    
    <div class="container">
        <div class="status-grid">
            <div class="status-card">
                <h3>🖥️ Server Information</h3>
                <div class="status-item">
                    <label>Server Software</label>
                    <value><?php echo htmlspecialchars($serverSoftware); ?></value>
                </div>
                <div class="status-item">
                    <label>Operating System</label>
                    <value><?php echo htmlspecialchars($serverOS); ?></value>
                </div>
                <div class="status-item">
                    <label>Server Time</label>
                    <value><?php echo $serverTime; ?></value>
                </div>
                <div class="status-item">
                    <label>Timezone</label>
                    <value><?php echo htmlspecialchars($timezone); ?></value>
                </div>
            </div>
            
            <div class="status-card">
                <h3>🐘 PHP Information</h3>
                <div class="status-item">
                    <label>PHP Version</label>
                    <value><?php echo $phpVersion; ?></value>
                </div>
                <div class="status-item">
                    <label>Memory Limit</label>
                    <value><?php echo htmlspecialchars($phpMemoryLimit); ?></value>
                </div>
                <div class="status-item">
                    <label>Max Upload Size</label>
                    <value><?php echo htmlspecialchars($phpMaxUpload); ?></value>
                </div>
                <div class="status-item">
                    <label>Max Post Size</label>
                    <value><?php echo htmlspecialchars($phpMaxPost); ?></value>
                </div>
                <div class="status-item">
                    <label>Max Execution Time</label>
                    <value><?php echo htmlspecialchars($phpMaxExecution); ?>s</value>
                </div>
            </div>
            
            <div class="status-card">
                <h3>💾 Database Status</h3>
                <div class="status-item">
                    <label>Status</label>
                    <span class="status-badge <?php echo $dbStatus === 'connected' ? 'success' : 'error'; ?>">
                        <?php echo strtoupper($dbStatus); ?>
                    </span>
                </div>
                <?php if ($dbStatus === 'connected'): ?>
                <div class="status-item">
                    <label>Version</label>
                    <value><?php echo htmlspecialchars($dbVersion); ?></value>
                </div>
                <div class="status-item">
                    <label>Database Name</label>
                    <value><?php echo DB_NAME; ?></value>
                </div>
                <div class="status-item">
                    <label>Database Size</label>
                    <value><?php echo number_format($dbSize, 2); ?> MB</value>
                </div>
                <?php else: ?>
                <div class="status-item">
                    <label>Error</label>
                    <value style="color: var(--accent-red); font-size: 0.75rem;"><?php echo htmlspecialchars($dbError); ?></value>
                </div>
                <?php endif; ?>
            </div>
            
            <div class="status-card">
                <h3>📦 PHP Extensions</h3>
                <?php foreach ($extensions as $ext => $loaded): ?>
                <div class="status-item">
                    <label><?php echo strtoupper($ext); ?></label>
                    <span class="status-badge <?php echo $loaded ? 'success' : 'error'; ?>">
                        <?php echo $loaded ? 'LOADED' : 'MISSING'; ?>
                    </span>
                </div>
                <?php endforeach; ?>
            </div>
            
            <div class="status-card">
                <h3>💿 Disk Space</h3>
                <div class="status-item">
                    <label>Total Space</label>
                    <value><?php echo formatBytes($diskSpace['total']); ?></value>
                </div>
                <div class="status-item">
                    <label>Used Space</label>
                    <value><?php echo formatBytes($diskSpace['used']); ?></value>
                </div>
                <div class="status-item">
                    <label>Free Space</label>
                    <value><?php echo formatBytes($diskSpace['free']); ?></value>
                </div>
                <div class="status-item">
                    <label>Usage</label>
                    <value><?php echo $diskSpace['percent']; ?>%</value>
                </div>
                <div class="progress-bar">
                    <div class="progress-fill <?php 
                        echo $diskSpace['percent'] > 90 ? 'danger' : ($diskSpace['percent'] > 70 ? 'warning' : ''); 
                    ?>" style="width: <?php echo $diskSpace['percent']; ?>%"></div>
                </div>
            </div>
            
            <div class="status-card">
                <h3>⚙️ System Configuration</h3>
                <div class="status-item">
                    <label>Admin Base URL</label>
                    <value style="font-size: 0.75rem; word-break: break-all;"><?php echo ADMIN_BASE_URL; ?></value>
                </div>
                <div class="status-item">
                    <label>Templates Directory</label>
                    <value style="font-size: 0.75rem;"><?php echo file_exists(TEMPLATES_DIR) ? '✓ Exists' : '✗ Missing'; ?></value>
                </div>
                <div class="status-item">
                    <label>Installers Directory</label>
                    <value style="font-size: 0.75rem;"><?php echo file_exists(INSTALLER_DIR) ? '✓ Exists' : '✗ Missing'; ?></value>
                </div>
                <div class="status-item">
                    <label>API Key Length</label>
                    <value><?php echo API_KEY_LENGTH; ?> characters</value>
                </div>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h2>Quick Actions</h2>
            </div>
            <div class="card-body" style="display: flex; gap: 1rem; flex-wrap: wrap;">
                <a href="index.php" class="btn btn-primary">Back to Dashboard</a>
                <a href="settings.php" class="btn btn-primary">Settings</a>
                <button onclick="location.reload()" class="btn btn-primary">Refresh Status</button>
            </div>
        </div>
    </div>
</body>
</html>

