<?php
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/config.php';
requireLogin();

$success = '';
$error = '';

// Get current admin info
$conn = getDBConnection();
$adminStmt = $conn->prepare("SELECT * FROM admin_users WHERE username = ?");
$adminStmt->bind_param("s", $_SESSION['admin_username']);
$adminStmt->execute();
$admin = $adminStmt->get_result()->fetch_assoc();

// Handle password change
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_password'])) {
    $currentPassword = $_POST['current_password'] ?? '';
    $newPassword = $_POST['new_password'] ?? '';
    $confirmPassword = $_POST['confirm_password'] ?? '';
    
    if (empty($currentPassword) || empty($newPassword) || empty($confirmPassword)) {
        $error = 'Please fill all fields';
    } elseif ($newPassword !== $confirmPassword) {
        $error = 'New passwords do not match';
    } elseif (strlen($newPassword) < 6) {
        $error = 'Password must be at least 6 characters';
    } elseif (!password_verify($currentPassword, $admin['password'])) {
        $error = 'Current password is incorrect';
    } else {
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        $updateStmt = $conn->prepare("UPDATE admin_users SET password = ? WHERE username = ?");
        $updateStmt->bind_param("ss", $hashedPassword, $_SESSION['admin_username']);
        if ($updateStmt->execute()) {
            $success = 'Password changed successfully';
        } else {
            $error = 'Failed to change password';
        }
    }
}

// Handle email update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_email'])) {
    $email = sanitizeInput($_POST['email'] ?? '');
    
    if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Invalid email address';
    } else {
        $updateStmt = $conn->prepare("UPDATE admin_users SET email = ? WHERE username = ?");
        $updateStmt->bind_param("ss", $email, $_SESSION['admin_username']);
        if ($updateStmt->execute()) {
            $success = 'Email updated successfully';
            $admin['email'] = $email;
        } else {
            $error = 'Failed to update email';
        }
    }
}

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - WHM Admin</title>
    <link rel="stylesheet" href="assets/css/admin.css">
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
</head>
<body>
    <div class="admin-header">
        <h1>Settings</h1>
        <div class="user-info">
            <a href="index.php" class="btn btn-primary btn-sm">Dashboard</a>
            <a href="logout.php" class="btn btn-danger btn-sm">Logout</a>
        </div>
    </div>
    
    <div class="container">
        <?php if ($success): ?>
        <div class="alert alert-success" style="margin-bottom: 1.5rem; padding: 1rem; background: rgba(46, 204, 113, 0.2); border: 1px solid rgba(46, 204, 113, 0.5); border-radius: 8px; color: #2ecc71;">
            <?php echo htmlspecialchars($success); ?>
        </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
        <div class="alert alert-danger" style="margin-bottom: 1.5rem; padding: 1rem; background: rgba(231, 76, 60, 0.2); border: 1px solid rgba(231, 76, 60, 0.5); border-radius: 8px; color: #e74c3c;">
            <?php echo htmlspecialchars($error); ?>
        </div>
        <?php endif; ?>
        
        <div class="card" style="margin-bottom: 1.5rem;">
            <div class="card-header">
                <h2>Change Password</h2>
            </div>
            <div class="card-body">
                <form method="POST">
                    <div class="form-group">
                        <label>Current Password</label>
                        <input type="password" name="current_password" class="form-control" required>
                    </div>
                    <div class="form-group">
                        <label>New Password</label>
                        <input type="password" name="new_password" class="form-control" required minlength="6">
                    </div>
                    <div class="form-group">
                        <label>Confirm New Password</label>
                        <input type="password" name="confirm_password" class="form-control" required minlength="6">
                    </div>
                    <button type="submit" name="change_password" class="btn btn-primary">Change Password</button>
                </form>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h2>Profile Information</h2>
            </div>
            <div class="card-body">
                <form method="POST">
                    <div class="form-group">
                        <label>Username</label>
                        <input type="text" value="<?php echo htmlspecialchars($admin['username']); ?>" class="form-control" disabled>
                    </div>
                    <div class="form-group">
                        <label>Email</label>
                        <input type="email" name="email" value="<?php echo htmlspecialchars($admin['email'] ?? ''); ?>" class="form-control">
                    </div>
                    <button type="submit" name="update_email" class="btn btn-primary">Update Email</button>
                </form>
            </div>
        </div>
        
        <div class="card" style="margin-top: 1.5rem;">
            <div class="card-header">
                <h2>System Information</h2>
            </div>
            <div class="card-body">
                <table class="table">
                    <tr>
                        <td><strong>PHP Version</strong></td>
                        <td><?php echo PHP_VERSION; ?></td>
                    </tr>
                    <tr>
                        <td><strong>Server Software</strong></td>
                        <td><?php echo $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown'; ?></td>
                    </tr>
                    <tr>
                        <td><strong>Admin Base URL</strong></td>
                        <td><?php echo ADMIN_BASE_URL; ?></td>
                    </tr>
                    <tr>
                        <td><strong>Timezone</strong></td>
                        <td><?php echo date_default_timezone_get(); ?></td>
                    </tr>
                </table>
            </div>
        </div>
    </div>
</body>
</html>

