<?php
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/config.php';
requireLogin();

// Delete client
if (isset($_GET['delete']) && isset($_GET['id'])) {
    $id = intval($_GET['id']);
    $conn = getDBConnection();
    $stmt = $conn->prepare("DELETE FROM clients WHERE id = ?");
    $stmt->bind_param("i", $id);
    if ($stmt->execute()) {
        header('Location: manage_clients.php?deleted=1');
        exit;
    }
    $conn->close();
}

// Regenerate API key
if (isset($_GET['regenerate']) && isset($_GET['id'])) {
    $id = intval($_GET['id']);
    $newApiKey = generateApiKey();
    $conn = getDBConnection();
    $stmt = $conn->prepare("UPDATE clients SET api_key = ? WHERE id = ?");
    $stmt->bind_param("si", $newApiKey, $id);
    if ($stmt->execute()) {
        header('Location: manage_clients.php?regenerated=1&new_key=' . urlencode($newApiKey));
        exit;
    }
    $conn->close();
}

$conn = getDBConnection();

// Search and filter
$search = sanitizeInput($_GET['search'] ?? '');
$statusFilter = sanitizeInput($_GET['status'] ?? 'all');
$page = max(1, intval($_GET['page'] ?? 1));
$perPage = 20;
$offset = ($page - 1) * $perPage;

// Build query
$where = [];
$params = [];
$types = '';

if (!empty($search)) {
    $where[] = "(domain LIKE ? OR api_key LIKE ?)";
    $searchParam = "%$search%";
    $params[] = $searchParam;
    $params[] = $searchParam;
    $types .= 'ss';
}

if ($statusFilter !== 'all') {
    $where[] = "status = ?";
    $params[] = $statusFilter;
    $types .= 's';
}

$whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

// Get total count
$countQuery = "SELECT COUNT(*) as total FROM clients $whereClause";
$countStmt = $conn->prepare($countQuery);
if (!empty($params)) {
    $countStmt->bind_param($types, ...$params);
}
$countStmt->execute();
$totalClients = $countStmt->get_result()->fetch_assoc()['total'];
$totalPages = ceil($totalClients / $perPage);

// Get clients
$query = "SELECT * FROM clients $whereClause ORDER BY created_at DESC LIMIT ? OFFSET ?";
$stmt = $conn->prepare($query);
if (!empty($params)) {
    $params[] = $perPage;
    $params[] = $offset;
    $types .= 'ii';
    $stmt->bind_param($types, ...$params);
} else {
    $stmt->bind_param("ii", $perPage, $offset);
}
$stmt->execute();
$clients = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Get statistics
$totalClientsAll = $conn->query("SELECT COUNT(*) as count FROM clients")->fetch_assoc()['count'];
$activeClients = $conn->query("SELECT COUNT(*) as count FROM clients WHERE status = 'active'")->fetch_assoc()['count'];
$pendingClients = $conn->query("SELECT COUNT(*) as count FROM clients WHERE status = 'pending'")->fetch_assoc()['count'];
$suspendedClients = $conn->query("SELECT COUNT(*) as count FROM clients WHERE status = 'suspended'")->fetch_assoc()['count'];

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Clients - WHM Admin</title>
    <link rel="stylesheet" href="assets/css/admin.css">
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <style>
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            flex-wrap: wrap;
            gap: 1rem;
        }
        .search-filter-bar {
            display: flex;
            gap: 1rem;
            margin-bottom: 2rem;
            flex-wrap: wrap;
            align-items: center;
        }
        .search-box {
            flex: 1;
            min-width: 250px;
            position: relative;
        }
        .search-box input {
            width: 100%;
            padding: 12px 16px 12px 40px;
            background: rgba(15, 15, 21, 0.7);
            border: 1px solid rgba(6, 182, 212, 0.3);
            border-radius: 8px;
            font-size: 14px;
            color: var(--text-primary);
            transition: all 0.3s;
        }
        .search-box input:focus {
            outline: none;
            border-color: var(--accent-blue);
            box-shadow: 0 0 0 3px rgba(6, 182, 212, 0.2), 0 0 15px rgba(6, 182, 212, 0.4);
        }
        .search-box::before {
            content: '🔍';
            position: absolute;
            left: 12px;
            top: 50%;
            transform: translateY(-50%);
            font-size: 16px;
        }
        .filter-select {
            padding: 12px 16px;
            background: rgba(15, 15, 21, 0.7);
            border: 1px solid rgba(6, 182, 212, 0.3);
            border-radius: 8px;
            font-size: 14px;
            color: var(--text-primary);
            cursor: pointer;
            transition: all 0.3s;
        }
        .filter-select:focus {
            outline: none;
            border-color: var(--accent-blue);
            box-shadow: 0 0 0 3px rgba(6, 182, 212, 0.2);
        }
        .stats-mini {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 1rem;
            margin-bottom: 2rem;
        }
        .stat-mini {
            background: rgba(21, 21, 32, 0.6);
            backdrop-filter: blur(10px);
            padding: 1rem;
            border-radius: 10px;
            border-left: 3px solid var(--accent-blue);
            text-align: center;
        }
        .stat-mini-value {
            font-size: 24px;
            font-weight: 700;
            color: var(--text-primary);
            margin-bottom: 0.25rem;
        }
        .stat-mini-label {
            font-size: 12px;
            color: var(--text-muted);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        .table-actions {
            display: flex;
            gap: 0.5rem;
            flex-wrap: wrap;
        }
        .pagination {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 0.5rem;
            margin-top: 2rem;
            flex-wrap: wrap;
        }
        .pagination a, .pagination span {
            padding: 8px 12px;
            background: rgba(21, 21, 32, 0.6);
            border: 1px solid rgba(6, 182, 212, 0.3);
            border-radius: 6px;
            color: var(--text-primary);
            text-decoration: none;
            font-size: 13px;
            transition: all 0.3s;
        }
        .pagination a:hover {
            background: rgba(6, 182, 212, 0.2);
            border-color: var(--accent-blue);
        }
        .pagination .current {
            background: linear-gradient(135deg, var(--accent-blue) 0%, var(--accent-teal) 100%);
            border-color: var(--accent-blue);
            color: white;
        }
        .btn-regenerate {
            background: linear-gradient(135deg, var(--accent-yellow) 0%, #ffcc00 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(255, 170, 0, 0.3);
        }
        .btn-regenerate:hover {
            box-shadow: 0 6px 25px rgba(255, 170, 0, 0.5);
        }
        .new-key-alert {
            background: rgba(20, 184, 166, 0.15);
            border: 1px solid rgba(20, 184, 166, 0.3);
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
        }
        .new-key-alert strong {
            color: var(--accent-green);
            display: block;
            margin-bottom: 0.5rem;
        }
        .new-key-alert code {
            background: rgba(15, 15, 21, 0.7);
            padding: 0.5rem 1rem;
            border-radius: 6px;
            font-size: 13px;
            word-break: break-all;
        }
    </style>
</head>
<body>
    <div class="admin-header">
        <h1>Manage Clients</h1>
        <div class="user-info">
            <a href="index.php" class="btn btn-primary btn-sm">Dashboard</a>
            <a href="add_client.php" class="btn btn-success btn-sm">Add Client</a>
            <span><?php echo htmlspecialchars($_SESSION['admin_username']); ?></span>
            <a href="logout.php" class="btn btn-danger btn-sm">Logout</a>
        </div>
    </div>
    
    <div class="container">
        <?php if (isset($_GET['deleted'])): ?>
            <div class="alert alert-success">Client deleted successfully!</div>
        <?php endif; ?>
        
        <?php if (isset($_GET['regenerated']) && isset($_GET['new_key'])): ?>
            <div class="new-key-alert">
                <strong>New API Key Generated!</strong>
                <code><?php echo htmlspecialchars($_GET['new_key']); ?></code>
                <p style="margin-top: 0.5rem; color: var(--text-secondary); font-size: 12px;">Copy this key and provide it to your client.</p>
            </div>
        <?php endif; ?>
        
        <div class="page-header">
            <div>
                <h1 style="font-size: 24px; margin-bottom: 0.25rem;">All Clients</h1>
                <p style="color: var(--text-muted); font-size: 14px;">Total: <?php echo $totalClientsAll; ?> clients</p>
            </div>
        </div>
        
        <div class="stats-mini">
            <div class="stat-mini">
                <div class="stat-mini-value"><?php echo $totalClientsAll; ?></div>
                <div class="stat-mini-label">Total</div>
            </div>
            <div class="stat-mini" style="border-left-color: var(--accent-green);">
                <div class="stat-mini-value"><?php echo $activeClients; ?></div>
                <div class="stat-mini-label">Active</div>
            </div>
            <div class="stat-mini" style="border-left-color: var(--accent-yellow);">
                <div class="stat-mini-value"><?php echo $pendingClients; ?></div>
                <div class="stat-mini-label">Pending</div>
            </div>
            <div class="stat-mini" style="border-left-color: var(--accent-red);">
                <div class="stat-mini-value"><?php echo $suspendedClients; ?></div>
                <div class="stat-mini-label">Suspended</div>
            </div>
        </div>
        
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1.5rem; gap: 1rem; flex-wrap: wrap;">
            <div style="display: flex; gap: 0.5rem; flex-wrap: wrap;">
                <a href="export_data.php?type=clients&format=csv" class="btn btn-primary btn-sm">Export CSV</a>
                <a href="export_data.php?type=clients&format=json" class="btn btn-primary btn-sm">Export JSON</a>
            </div>
        </div>
        
        <div class="search-filter-bar">
            <form method="GET" class="search-box" style="display: flex; gap: 0.5rem;">
                <input type="text" name="search" placeholder="Search by domain or API key..." value="<?php echo htmlspecialchars($search); ?>">
                <select name="status" class="filter-select" style="width: auto; min-width: 150px;" onchange="this.form.submit()">
                    <option value="all" <?php echo $statusFilter === 'all' ? 'selected' : ''; ?>>All Status</option>
                    <option value="active" <?php echo $statusFilter === 'active' ? 'selected' : ''; ?>>Active</option>
                    <option value="pending" <?php echo $statusFilter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                    <option value="suspended" <?php echo $statusFilter === 'suspended' ? 'selected' : ''; ?>>Suspended</option>
                </select>
                <button type="submit" class="btn btn-primary" style="white-space: nowrap;">Search</button>
                <?php if (!empty($search) || $statusFilter !== 'all'): ?>
                    <a href="manage_clients.php" class="btn btn-sm" style="background: rgba(255, 51, 102, 0.15); border: 1px solid rgba(255, 51, 102, 0.3); color: var(--accent-red);">Clear</a>
                <?php endif; ?>
            </form>
        </div>
        
        <div class="card">
            <div class="table-wrapper">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Domain</th>
                            <th>Template</th>
                            <th>API Key</th>
                            <th>Status</th>
                            <th>Created</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($clients)): ?>
                        <tr>
                            <td colspan="6" class="empty-state">
                                <div class="empty-state-icon">📭</div>
                                <p>No clients found.</p>
                            </td>
                        </tr>
                        <?php else: ?>
                        <?php foreach ($clients as $client): ?>
                        <tr>
                            <td>
                                <strong><?php echo htmlspecialchars($client['domain']); ?></strong>
                            </td>
                            <td>
                                <span class="badge badge-active"><?php echo htmlspecialchars($client['template_selected']); ?></span>
                            </td>
                            <td>
                                <div style="display: flex; align-items: center; gap: 8px; flex-wrap: wrap;">
                                    <code style="font-size: 10px; max-width: 200px; overflow: hidden; text-overflow: ellipsis; display: inline-block; font-family: 'Courier New', monospace;">
                                        <?php echo htmlspecialchars($client['api_key']); ?>
                                    </code>
                                    <button class="btn btn-sm" onclick="copyToClipboard('<?php echo htmlspecialchars($client['api_key']); ?>')" title="Copy" style="padding: 5px 12px; font-size: 11px; background: rgba(6, 182, 212, 0.15); border: 1px solid rgba(6, 182, 212, 0.4); color: var(--accent-blue);">Copy</button>
                                </div>
                            </td>
                            <td>
                                <span class="badge badge-<?php echo $client['status']; ?>">
                                    <?php echo ucfirst($client['status']); ?>
                                </span>
                            </td>
                            <td style="color: var(--text-secondary); font-size: 12px;">
                                <?php echo date('M d, Y H:i', strtotime($client['created_at'])); ?>
                            </td>
                            <td>
                                <div class="table-actions">
                                    <a href="edit_client.php?id=<?php echo $client['id']; ?>" class="btn btn-primary btn-sm">Edit</a>
                                    <a href="?regenerate=1&id=<?php echo $client['id']; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo $statusFilter !== 'all' ? '&status=' . urlencode($statusFilter) : ''; ?>" class="btn btn-regenerate btn-sm" onclick="return confirm('Regenerate API key for <?php echo htmlspecialchars($client['domain']); ?>?')">Regenerate</a>
                                    <a href="?delete=1&id=<?php echo $client['id']; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo $statusFilter !== 'all' ? '&status=' . urlencode($statusFilter) : ''; ?>" class="btn btn-danger btn-sm" onclick="return confirm('Delete client <?php echo htmlspecialchars($client['domain']); ?>? This action cannot be undone!')">Delete</a>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            
            <?php if ($totalPages > 1): ?>
            <div class="pagination">
                <?php if ($page > 1): ?>
                    <a href="?page=<?php echo $page - 1; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo $statusFilter !== 'all' ? '&status=' . urlencode($statusFilter) : ''; ?>">« Previous</a>
                <?php endif; ?>
                
                <?php
                $startPage = max(1, $page - 2);
                $endPage = min($totalPages, $page + 2);
                for ($i = $startPage; $i <= $endPage; $i++):
                ?>
                    <?php if ($i == $page): ?>
                        <span class="current"><?php echo $i; ?></span>
                    <?php else: ?>
                        <a href="?page=<?php echo $i; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo $statusFilter !== 'all' ? '&status=' . urlencode($statusFilter) : ''; ?>"><?php echo $i; ?></a>
                    <?php endif; ?>
                <?php endfor; ?>
                
                <?php if ($page < $totalPages): ?>
                    <a href="?page=<?php echo $page + 1; ?><?php echo !empty($search) ? '&search=' . urlencode($search) : ''; ?><?php echo $statusFilter !== 'all' ? '&status=' . urlencode($statusFilter) : ''; ?>">Next »</a>
                <?php endif; ?>
            </div>
            <?php endif; ?>
        </div>
    </div>
    
    <script>
        function copyToClipboard(text) {
            navigator.clipboard.writeText(text).then(function() {
                var btn = event.target;
                var original = btn.textContent;
                btn.textContent = 'Copied!';
                btn.style.background = 'rgba(20, 184, 166, 0.3)';
                setTimeout(function() {
                    btn.textContent = original;
                    btn.style.background = '';
                }, 1500);
            });
        }
    </script>
</body>
</html>

