<?php
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/config.php';
requireLogin();

$conn = getDBConnection();

$search = sanitizeInput($_GET['search'] ?? '');
$statusFilter = sanitizeInput($_GET['status'] ?? 'all');
$page = max(1, intval($_GET['page'] ?? 1));
$perPage = 20;
$offset = ($page - 1) * $perPage;

$where = [];
$params = [];
$types = '';

if (!empty($search)) {
    $where[] = "(c.domain LIKE ? OR i.api_key LIKE ? OR t.name LIKE ?)";
    $searchParam = "%$search%";
    $params[] = $searchParam;
    $params[] = $searchParam;
    $params[] = $searchParam;
    $types .= 'sss';
}

if ($statusFilter !== 'all') {
    $where[] = "i.install_status = ?";
    $params[] = $statusFilter;
    $types .= 's';
}

$whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

$countQuery = "SELECT COUNT(*) as total FROM installations i
               LEFT JOIN clients c ON i.client_id = c.id
               LEFT JOIN templates t ON i.template_id = t.id
               $whereClause";
$countStmt = $conn->prepare($countQuery);
if (!empty($params)) {
    $countStmt->bind_param($types, ...$params);
}
$countStmt->execute();
$totalCount = $countStmt->get_result()->fetch_assoc()['total'];
$totalPages = ceil($totalCount / $perPage);

$query = "SELECT i.*, c.domain, c.template_selected, t.name as template_name 
          FROM installations i
          LEFT JOIN clients c ON i.client_id = c.id
          LEFT JOIN templates t ON i.template_id = t.id
          $whereClause
          ORDER BY i.install_date DESC 
          LIMIT ? OFFSET ?";
$stmt = $conn->prepare($query);
if (!empty($params)) {
    $params[] = $perPage;
    $params[] = $offset;
    $types .= 'ii';
    $stmt->bind_param($types, ...$params);
} else {
    $stmt->bind_param("ii", $perPage, $offset);
}
$stmt->execute();
$installations = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Get total count
$totalCount = $conn->query("SELECT COUNT(*) as count FROM installations")->fetch_assoc()['count'];
$totalPages = ceil($totalCount / $perPage);

// Get statistics
$totalInstalls = $conn->query("SELECT COUNT(*) as count FROM installations")->fetch_assoc()['count'];
$completedInstalls = $conn->query("SELECT COUNT(*) as count FROM installations WHERE install_status = 'completed'")->fetch_assoc()['count'];
$failedInstalls = $conn->query("SELECT COUNT(*) as count FROM installations WHERE install_status = 'failed'")->fetch_assoc()['count'];
$pendingInstalls = $conn->query("SELECT COUNT(*) as count FROM installations WHERE install_status IN ('pending', 'downloading', 'extracting')")->fetch_assoc()['count'];

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Installation History - WHM Admin</title>
    <link rel="stylesheet" href="assets/css/admin.css">
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
</head>
<body>
    <div class="admin-header">
        <h1>Installation History</h1>
        <div class="user-info">
            <a href="index.php" class="btn btn-primary btn-sm">Dashboard</a>
            <a href="logout.php" class="btn btn-danger btn-sm">Logout</a>
        </div>
    </div>
    
    <div class="container">
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1.5rem; gap: 1rem; flex-wrap: wrap;">
            <div style="display: flex; gap: 0.5rem; flex-wrap: wrap;">
                <a href="export_data.php?type=installations&format=csv" class="btn btn-primary btn-sm">Export CSV</a>
                <a href="export_data.php?type=installations&format=json" class="btn btn-primary btn-sm">Export JSON</a>
            </div>
        </div>
        
        <div class="card" style="margin-bottom: 1.5rem;">
            <div class="card-body">
                <form method="GET" style="display: flex; gap: 1rem; flex-wrap: wrap; align-items: end;">
                    <div style="flex: 1; min-width: 200px;">
                        <label style="display: block; margin-bottom: 0.5rem; color: var(--text-secondary); font-size: 0.875rem;">Search</label>
                        <input type="text" name="search" value="<?php echo htmlspecialchars($search); ?>" placeholder="Domain, API Key, Template..." class="form-control" style="width: 100%;">
                    </div>
                    <div style="min-width: 150px;">
                        <label style="display: block; margin-bottom: 0.5rem; color: var(--text-secondary); font-size: 0.875rem;">Status</label>
                        <select name="status" class="form-control" style="width: 100%;">
                            <option value="all" <?php echo $statusFilter === 'all' ? 'selected' : ''; ?>>All Status</option>
                            <option value="pending" <?php echo $statusFilter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                            <option value="downloading" <?php echo $statusFilter === 'downloading' ? 'selected' : ''; ?>>Downloading</option>
                            <option value="extracting" <?php echo $statusFilter === 'extracting' ? 'selected' : ''; ?>>Extracting</option>
                            <option value="completed" <?php echo $statusFilter === 'completed' ? 'selected' : ''; ?>>Completed</option>
                            <option value="failed" <?php echo $statusFilter === 'failed' ? 'selected' : ''; ?>>Failed</option>
                        </select>
                    </div>
                    <div>
                        <button type="submit" class="btn btn-primary">Filter</button>
                        <a href="installation_history.php" class="btn btn-primary" style="margin-left: 0.5rem;">Reset</a>
                    </div>
                </form>
            </div>
        </div>
        
        <div class="stats-grid" style="margin-bottom: 2rem;">
            <div class="stat-card visitors">
                <div class="stat-card-header">
                    <span class="stat-card-title">Total Installations</span>
                    <div class="stat-card-icon">📥</div>
                </div>
                <div class="stat-card-value"><?php echo $totalInstalls; ?></div>
            </div>
            
            <div class="stat-card logins">
                <div class="stat-card-header">
                    <span class="stat-card-title">Completed</span>
                    <div class="stat-card-icon">✓</div>
                </div>
                <div class="stat-card-value"><?php echo $completedInstalls; ?></div>
            </div>
            
            <div class="stat-card cards">
                <div class="stat-card-header">
                    <span class="stat-card-title">Failed</span>
                    <div class="stat-card-icon">✗</div>
                </div>
                <div class="stat-card-value"><?php echo $failedInstalls; ?></div>
            </div>
            
            <div class="stat-card blocked">
                <div class="stat-card-header">
                    <span class="stat-card-title">In Progress</span>
                    <div class="stat-card-icon">⏱</div>
                </div>
                <div class="stat-card-value"><?php echo $pendingInstalls; ?></div>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h2>Installation Logs</h2>
            </div>
            <div class="table-wrapper">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Date & Time</th>
                            <th>Domain</th>
                            <th>Template</th>
                            <th>Status</th>
                            <th>Error Message</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($installations)): ?>
                        <tr>
                            <td colspan="5" class="empty-state">
                                <div class="empty-state-icon">📭</div>
                                <p>No installation history yet.</p>
                            </td>
                        </tr>
                        <?php else: ?>
                        <?php foreach ($installations as $install): ?>
                        <tr>
                            <td style="color: var(--text-secondary); font-size: 12px;">
                                <?php echo date('M d, Y H:i:s', strtotime($install['install_date'])); ?>
                            </td>
                            <td>
                                <strong><?php echo htmlspecialchars($install['domain'] ?? 'N/A'); ?></strong>
                            </td>
                            <td>
                                <span class="badge badge-active">
                                    <?php echo htmlspecialchars($install['template_name'] ?? $install['template_selected'] ?? 'N/A'); ?>
                                </span>
                            </td>
                            <td>
                                <span class="badge badge-<?php 
                                    echo $install['install_status'] === 'completed' ? 'active' : 
                                        ($install['install_status'] === 'failed' ? 'danger' : 'warning'); 
                                ?>">
                                    <?php echo ucfirst($install['install_status']); ?>
                                </span>
                            </td>
                            <td style="color: var(--text-secondary); font-size: 11px; max-width: 300px;">
                                <?php echo htmlspecialchars($install['error_message'] ?? '-'); ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            
            <?php if ($totalPages > 1): ?>
            <div class="pagination" style="padding: 1.5rem; display: flex; justify-content: center; gap: 10px;">
                <?php if ($page > 1): ?>
                <a href="?page=<?php echo $page - 1; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo urlencode($statusFilter); ?>" class="btn btn-primary btn-sm">Previous</a>
                <?php endif; ?>
                
                <span style="padding: 8px 16px; color: var(--text-secondary);">
                    Page <?php echo $page; ?> of <?php echo $totalPages; ?> (Total: <?php echo $totalCount; ?>)
                </span>
                
                <?php if ($page < $totalPages): ?>
                <a href="?page=<?php echo $page + 1; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo urlencode($statusFilter); ?>" class="btn btn-primary btn-sm">Next</a>
                <?php endif; ?>
            </div>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>

