<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';
require_once '../config/config.php';

// Rate limiting (simple implementation)
$rateLimitFile = sys_get_temp_dir() . '/api_rate_limit_' . md5($_SERVER['REMOTE_ADDR']);
$currentTime = time();
$requests = [];

if (file_exists($rateLimitFile)) {
    $requests = json_decode(file_get_contents($rateLimitFile), true) ?: [];
    // Remove requests older than 1 minute
    $requests = array_filter($requests, function($time) use ($currentTime) {
        return ($currentTime - $time) < 60;
    });
}

if (count($requests) >= RATE_LIMIT_REQUESTS) {
    http_response_code(429);
    echo json_encode(['error' => 'Rate limit exceeded']);
    exit;
}

$requests[] = $currentTime;
file_put_contents($rateLimitFile, json_encode($requests));

// Get input
$input = json_decode(file_get_contents('php://input'), true);
$apiKey = $input['api_key'] ?? '';
$domain = $input['domain'] ?? '';

if (empty($apiKey) || empty($domain)) {
    http_response_code(400);
    echo json_encode(['valid' => false, 'error' => 'Missing parameters']);
    exit;
}

$conn = getDBConnection();

// Validate API key and domain
$stmt = $conn->prepare("SELECT id, domain, template_selected, status FROM clients WHERE api_key = ? AND domain = ?");
$stmt->bind_param("ss", $apiKey, $domain);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 1) {
    $client = $result->fetch_assoc();
    
    // Check if template exists
    $templateCheck = $conn->prepare("SELECT zip_file_path FROM templates WHERE folder_name = ? AND is_active = 1");
    $templateCheck->bind_param("s", $client['template_selected']);
    $templateCheck->execute();
    $templateResult = $templateCheck->get_result();
    
    if ($templateResult->num_rows === 1) {
        $template = $templateResult->fetch_assoc();
        
        // Log installation attempt
        $logStmt = $conn->prepare("INSERT INTO installations (client_id, api_key, template_id, install_status) VALUES (?, ?, (SELECT id FROM templates WHERE folder_name = ?), 'pending')");
        $logStmt->bind_param("iss", $client['id'], $apiKey, $client['template_selected']);
        $logStmt->execute();
        
        // Generate download URL with token
        $token = bin2hex(random_bytes(16));
        $_SESSION['download_token_' . $token] = [
            'api_key' => $apiKey,
            'domain' => $domain,
            'expires' => time() + 3600 // 1 hour expiry
        ];
        
        echo json_encode([
            'valid' => true,
            'template' => $client['template_selected'],
            'download_url' => ADMIN_BASE_URL . '/api/download_template.php?token=' . $token,
            'status' => $client['status']
        ]);
    } else {
        http_response_code(404);
        echo json_encode(['valid' => false, 'error' => 'Template not found']);
    }
} else {
    http_response_code(403);
    echo json_encode(['valid' => false, 'error' => 'Invalid API key or domain']);
}

$conn->close();
?>

