<?php
require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/config.php';
requireLogin();

$success = '';
$error = '';
$apiKey = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $domain = sanitizeInput($_POST['domain'] ?? '');
    $template = sanitizeInput($_POST['template'] ?? '');
    
    if (empty($domain) || empty($template)) {
        $error = 'Please fill all fields';
    } else {
        $conn = getDBConnection();
        
        // Check if domain already exists
        $check = $conn->prepare("SELECT id FROM clients WHERE domain = ?");
        $check->bind_param("s", $domain);
        $check->execute();
        if ($check->get_result()->num_rows > 0) {
            $error = 'Domain already registered';
        } else {
            // Generate API key
            $apiKey = generateApiKey();
            
            // Insert client
            $stmt = $conn->prepare("INSERT INTO clients (domain, api_key, template_selected, status) VALUES (?, ?, ?, 'pending')");
            $stmt->bind_param("sss", $domain, $apiKey, $template);
            
            if ($stmt->execute()) {
                $success = "Client added successfully!";
            } else {
                $error = 'Failed to add client';
            }
        }
        
        $conn->close();
    }
}

// Get available templates
$conn = getDBConnection();
$templates = $conn->query("SELECT * FROM templates WHERE is_active = 1")->fetch_all(MYSQLI_ASSOC);
$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add New Client - WHM Admin</title>
    <link rel="stylesheet" href="assets/css/admin.css">
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <style>
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
        }
        .api-key-display {
            background: rgba(0, 212, 255, 0.05);
            backdrop-filter: blur(10px);
            border: 2px solid var(--accent-blue);
            color: white;
            padding: 1.75rem;
            border-radius: 12px;
            margin: 1.75rem 0;
            box-shadow: 0 0 30px rgba(0, 212, 255, 0.2);
            position: relative;
            overflow: hidden;
        }
        .api-key-display::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(0, 212, 255, 0.1), transparent);
            animation: shine 3s infinite;
        }
        @keyframes shine {
            0% { left: -100%; }
            100% { left: 100%; }
        }
        .api-key-display h3 {
            color: var(--accent-blue);
            margin-bottom: 0.75rem;
            font-size: 0.875rem;
            text-transform: uppercase;
            letter-spacing: 0.05em;
            font-weight: 600;
            text-shadow: 0 0 10px rgba(0, 212, 255, 0.3);
        }
        .api-key-value {
            font-family: 'Courier New', monospace;
            font-size: 0.9rem;
            word-break: break-all;
            background: rgba(0, 212, 255, 0.1);
            padding: 1.25rem;
            border-radius: 8px;
            margin: 0.75rem 0;
            border: 1px solid rgba(0, 212, 255, 0.2);
            color: #fff;
            box-shadow: 0 0 15px rgba(0, 212, 255, 0.1);
        }
        .copy-api-btn {
            background: linear-gradient(135deg, var(--accent-blue) 0%, #0099cc 100%);
            color: white;
            border: none;
            padding: 0.625rem 1.25rem;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 500;
            margin-top: 0.75rem;
            transition: all 0.3s;
            font-size: 13px;
            box-shadow: 0 4px 15px rgba(0, 212, 255, 0.3);
        }
        .copy-api-btn:hover {
            background: linear-gradient(135deg, #00b8e6 0%, var(--accent-blue) 100%);
            transform: translateY(-2px);
            box-shadow: 0 6px 25px rgba(0, 212, 255, 0.5);
        }
        .info-box {
            background: rgba(0, 212, 255, 0.05);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(0, 212, 255, 0.2);
            border-left: 4px solid var(--accent-blue);
            padding: 1.5rem;
            border-radius: 12px;
            margin-top: 1.75rem;
            box-shadow: 0 0 20px rgba(0, 212, 255, 0.1);
        }
        .info-box h4 {
            color: #fff;
            margin-bottom: 0.75rem;
            font-size: 0.9375rem;
            font-weight: 600;
        }
        .info-box ol {
            margin-left: 1.5rem;
            line-height: 2;
            color: #b0b0c0;
        }
        .info-box li {
            margin-bottom: 0.5rem;
        }
        .info-box a {
            color: var(--accent-blue);
            text-shadow: 0 0 10px rgba(0, 212, 255, 0.3);
        }
    </style>
</head>
<body>
    <div class="admin-header">
        <h1>Add New Client</h1>
        <div class="user-info">
            <a href="index.php" class="btn btn-primary btn-sm">Back to Dashboard</a>
        </div>
    </div>
    
    <div class="container">
        <div class="card" style="max-width: 800px; margin: 0 auto;">
            <?php if ($success): ?>
                <div class="alert alert-success">
                    <span>✅</span>
                    <span><?php echo $success; ?></span>
                </div>
                
                <?php if ($apiKey): ?>
                <div class="api-key-display">
                    <h3>API Key Generated</h3>
                    <div class="api-key-value" id="apiKeyValue"><?php echo htmlspecialchars($apiKey); ?></div>
                    <button class="copy-api-btn" onclick="copyApiKey()">Copy API Key</button>
                </div>
                <?php endif; ?>
                
                <div class="info-box">
                    <h4>Next Steps:</h4>
                    <ol>
                        <li>Copy the <strong>API Key</strong> above</li>
                        <li>Download <strong>installer.php</strong> from <a href="download_installer.php" target="_blank" style="color: var(--primary);">here</a></li>
                        <li>Send both to your client</li>
                        <li>Client uploads installer.php to their server root directory</li>
                        <li>Client opens: <code>https://<?php echo htmlspecialchars($domain ?? 'client-domain.com'); ?>/installer.php</code></li>
                        <li>Client enters the API Key</li>
                        <li>Template will automatically install!</li>
                    </ol>
                </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="alert alert-error">
                    <span>❌</span>
                    <span><?php echo htmlspecialchars($error); ?></span>
                </div>
            <?php endif; ?>
            
            <form method="POST">
                <div class="form-group">
                    <label class="form-label">Domain</label>
                    <input type="text" name="domain" class="form-control" placeholder="example.com" required>
                    <small style="color: var(--gray); font-size: 0.8125rem; margin-top: 0.25rem; display: block;">
                        Enter the client's domain (without http:// or https://)
                    </small>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Template</label>
                    <select name="template" class="form-control" required>
                        <option value="">Select a template...</option>
                        <?php foreach ($templates as $template): ?>
                            <option value="<?php echo htmlspecialchars($template['folder_name']); ?>">
                                <?php echo htmlspecialchars($template['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <?php if (empty($templates)): ?>
                        <small style="color: var(--danger); font-size: 0.8125rem; margin-top: 0.25rem; display: block;">
                            ⚠️ No templates available. <a href="manage_templates.php" style="color: var(--primary);">Add a template first</a>
                        </small>
                    <?php endif; ?>
                </div>
                
                <button type="submit" class="btn btn-primary" style="width: 100%; padding: 0.875rem;">
                    Generate API Key & Add Client
                </button>
            </form>
        </div>
    </div>
    
    <script>
        function copyApiKey() {
            const apiKey = document.getElementById('apiKeyValue').textContent;
            navigator.clipboard.writeText(apiKey).then(function() {
                const btn = event.target;
                const originalText = btn.textContent;
                btn.textContent = '✅ Copied!';
                btn.style.background = '#10b981';
                btn.style.color = 'white';
                setTimeout(() => {
                    btn.textContent = originalText;
                    btn.style.background = 'white';
                    btn.style.color = 'var(--primary)';
                }, 2000);
            });
        }
    </script>
</body>
</html>
